// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Grid_XY.fsh                                    ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.2 (2017/01/30)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To generate a regular cartesian coordinates grid.             ==
// ==                                                                        ==
// == Description: This shader creates a two dimensional grid of lines or of ==
// == points. It is primarily for use as a scene development tool, using its ==
// == output as a reference grid for positioning scene elements.             == 
// ==                                                                        ==
// == Note,  while this shader is not intended for "artistic effects" use it ==
// == may occasionaly be useful in that role.  However, if some variation on ==
// == the grid theme is required it should not be added as an option to this ==
// == shader, which should retain it development tool like nature, but a new ==
// == special effects style shader should be produced from it instead.       ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2016/01/28 - v1.0.0 - Initial version                                ==
// ==   2016/01/29 - v1.0.1 - Rewritten                                      ==
// ==   2016/01/30 - v1.0.2 - Added fuzz and transparency options            ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// EmuLib standard scale and hotspot parameters.  Note, a hotspot of (0.5,0.5)
// may be required to produce a symetric result.

uniform vec2 Emu_Grid_XY_scale;
uniform vec2 Emu_Grid_XY_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Grid_XY_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Grid_XY_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The shader's mode is controlled by a signed two  decimal  digit  parameter.
// This encodes the following information:
//
//   Grid inversion    - controlled by the sign of Emu_Grid_XY_mode
//   Grid spacing mode - controlled by the more significant digit
//   Grid display mode - controlled by the less significant digit
//
// The grid is displayed as either an array of dots at the grid points or as a
// set lines connecting the grid points depending on the grid display mode. It 
// is specified by the less significant decimal digit of the mode and may take
// the following values 
//
//    0 - Use the default value of 1
//    1 - Lines connecting grid points.
//    2 - Array of dots at grid points.
//
// with a value of 0 being treated as 1.  When  mode  is negated then the grid
// is inverted with the grid lines or points being transparent while the areas
// between are rendered in what would normally have been the grid colour.
//
// The manner in which the grid spacing is specified is controlled by the more
// significant digit of the mode. This may be
//
//    0 - Use the default value of 1
//    1 - Emu_Grid_XY_size specifies the number of grid cells.
//    2 - Emu_Grid_XY_size specifies the absolute grid spacing.
//
// The default mode, 0, will cause the grid to be displayed as a set of lines
// with its size being specified in terms of grid cells.

uniform int Emu_Grid_XY_mode;

bool invert   = Emu_Grid_XY_mode < 0;
int base_mode = abs(Emu_Grid_XY_mode);

int size_mode = EMU_DEFAULT ( int, int(mod(base_mode/10.0,10.0)), 1 );
int show_mode = EMU_DEFAULT ( int, int(mod(base_mode/1.00,10.0)), 1 );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The number of points or lines in the grid is specified by the size  control
// parameter which, depending on the size mode, either specifies the number of
// cells the window is to be divided into or the dimensions of the cells.  The
// default is to use a single cell.

uniform vec2 Emu_Grid_XY_size;

vec2 grid_x_02 = (size_mode==1) ? vec2(1.0) : u_WindowSize;

vec2 grid_size = EMU_DEFAULT ( vec2, Emu_Grid_XY_size, grid_x_02 );
vec2 grid_cell = (size_mode==2) ? grid_size : u_WindowSize / grid_size;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the grid lines are parallel to the X and Y axes, however this
// can be modified by using the angle, skew and taper control parameters.
//
//    angle - rotates the grid by the specified angle in degrees
//    taper - tapers the grid by the specified factor
//    skew  - skews the grid by the specified factor
//
// A skew of (1.0,0.0) will leave the horizontal grid lines unchanged while
// causing the vertical grid lines to slope by 45 degrees. A positive taper
// causes the grid line spacing to narrow in the direction of the axis that
// the taper has been applied to. Skew and taper may be any value but it is
// expected that they will normaly be in the range -1.0 to +1.0.
//
// Note, the rotation is centered on the specified hotspot, see above. Also
// note,  that in order to obtain a symetric tapering a hotspot position of
// 0.5 in one or both directions needs to be specified.
//
// The defaults for all of these parameters are zero.

uniform vec2  Emu_Grid_XY_skew;
uniform vec2  Emu_Grid_XY_taper;
uniform float Emu_Grid_XY_angle;

#define skew  Emu_Grid_XY_skew
#define taper Emu_Grid_XY_taper

float angle = radians(Emu_Grid_XY_angle);

mat2 rotation = mat2 ( cos(angle), sin(angle), 
                      -sin(angle), cos(angle)
                     );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The default thickness of the grid lines or points has been chosen so  that
// they should be easily visible for the normal range of window sizes, but it
// will often then too heavy for uses other than scene development.  However, 
// it can be overridden using the grid weight control  parameter.  Specifying 
// a value larger than 1.0 for this parameter will increase the thickness and
// values less than 1.0 will decrease it.  The default value is (1.0,1.0), it
// is therefore possible to eliminate one set of grid lines  by  setting  the
// corresponding component of weight to zero - but not both as (0.0,0.0) will
// be interpreted as not being specified and replaced by the default value.

uniform vec2 Emu_Grid_XY_weight;

vec2 weight = EMU_DEFAULT ( vec2, Emu_Grid_XY_weight, vec2(1.0) ) * 4.0;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the grid lines will have sharp edges.  However, this can result
// in those edges having jagged edges when the image is rendered. This effect
// can be reduced by softening the edges which is controlled by the following
// control parameter. A fuzz value of 0.0 leaves the edges sharp while larger
// values introduce softening. For most purposes a values between 0.5 and 2.0
// produce good results. Larger values can be used can be used but these tend
// to produce "artistic" effects rather than being purely utilitarian.  Note,
// the fuzz is added to the outside of grid line therefore when using fuzz it
// will commonly be the case that the line weight needs to be reduced to keep
// the same visible line width that was seen without fuzz. 

uniform vec2 Emu_Grid_XY_fuzz;
#define fuzz Emu_Grid_XY_fuzz

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the grid will be an opaque black but this may be overriden with
// the following pair of control parameters. Note, the output from the shader
// is modified by gl_Color which can be controlled with the normal color: and
// opacity clauses in the scene file.

uniform vec3  Emu_Grid_XY_colour;
uniform float Emu_Grid_XY_transparency;

#define colour  Emu_Grid_XY_colour 
#define opacity ( 1.0 - Emu_Grid_XY_transparency ) 
 
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the grid is always displayed, however, in order to increase its
// visibility it is sometimes usefull to be able to make it blink.  The blink
// parameter is used to specify the blink rate in Hertz.

uniform float Emu_Grid_XY_blink;
#define blink Emu_Grid_XY_blink

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 { 
   // Get the coordinates of the current point.

   vec2 xy = gl_FragCoord.xy;

   xy = ( xy - hotspot*u_WindowSize ) / scale;

   // Apply any requested rotation, skew or taper.

   xy *= rotation;
   xy += xy.yx * skew;
   xy /= 1.0 - taper*(xy/u_WindowSize).yx;

   // Get the nearest point that is on a grid line.

   vec2 gg = round(xy/grid_cell) * grid_cell;

   // Determine if current point is on a grid line or not.
   // Using smoothstep produces a softer edge to the grid
   // line and reduces "jaggies" due to aliasing effects.

   vec2 ff = weight*fuzz;

   vec2 dd = 1.0 - smoothstep ( weight, weight+ff, abs(xy-gg) );

   // Depending on the mode either points on grid lines or 
   // only those on grid points should be rendered opaque.

   float aa = 0.0;

   switch ( abs(show_mode) )
    { case 1: aa = max ( dd.x, dd.y );  break;
      case 2: aa = min ( dd.x, dd.y );  break;
    }

   // Optionaly invert the grid pattern.

   if ( invert ) aa = 1.0 - aa;

   // Optionaly make the grid blink.

   aa *= step ( 0.5, 1.0 - fract(u_Elapsed*blink) );

   // Set the colour and opacity of the current pixel.

   gl_FragColor = vec4(colour,aa*opacity) * gl_Color;

}